//___________________________________
//                                   \
// CLOCK Engine.  version 1.0
//___________________________________/


/*
 Based on the Game clock object by WIP.

 Usage:
   RequireScript("clockengine.js");
   var Clock = new ClockEngine( OpenFile('slot1.sav').read('MsTime',0) );
   var TimeString=Clock.getTimeStr();
   GetSystemFont().drawText(20,10, " This game has been running for "+TimeString);
   FlipScreen();

   Known Bugs:	To make it backwards compatible with WIP's systemscript, 
		after RequireScript(), add this line to your code: Clock=ClockEngine;

		Functions in this library:

*******************************************************************************
ClockEngine(oldMsTime)
	Create a new ClockEngine object. The oldMsTime is the milliseconds the game
	already has been running (from a savestate). This parameter is optional,
	if not given, it will assume the game has just started.
	example:
		var ClockPlayer1 = new ClockEngine();

start(oldMsTime)
	(Re)sets the running time of the game. The parameter is optional.
	if not given, it will assume the game has just started.
	example:
		Clock.start();

pause(redefineGetTime)
unpause(redefineGetTime)
	(un)Pauses the running time of the game.
	If no parameter is given, it will use the default value of redefineGetTime (true by default),
	which will modify GetTime() so the counter stops ticking.
	If you still need the real function GetTime(), then use GetTime_()
	example:
		Clock.pause(true); //pause is now on, time has stopped

isPaused()
	Boolean. True if pause is on.
	example:
		if( Clock.isPause() ) { Clock.pause(false); }

getTimeStr(returnAnArray,spheretime)
	String/Array. Returns an array of 3 elements: [hours, mins, secs] if returnAnArray is true.
	Else it returns a string containing the running time (in hhh:mm:ss).
	if Spheretime is true, it will show the time the Sphere Engine has been running, if false,
	it will show the amount of time the game has been running (which is affected by pause() )
	getTimeStr() will also set the internal strings: .seconds .minutes .hours
	example:
		var TimeArray=Clock.getTimeStr(true,false);
		font.drawText(10, 10, TimeArray[0]); // Hours, same as Clock.hours
		font.drawText(10, 20, Clock.minutes); // Minutes, same as TimeArray[1]
		font.drawText(10, 30, TimeArray[2]); // Seconds, same as Clock.seconds

getMs()
	Returns the delta in milliseconds a game has been running, this value can be saved and used
	to .start() the clock again.
	example:
		var sav = OpenFile('slot1.sav'); 
		sav.write('MsTime',Clock.getMs());
		sav.flush();
		GarbageCollect(); //cleans up unused memory


	Hacks:
		you can override the original GetTime(), so all your game functions that rely on GetTime() will work
		perfectly with pause (will make things slower though, so you may consider using frames instead of milliseconds to synch):

		RequireScript("clockengine.js");
		var Clock = new ClockEngine();
	And then:
		GetTime = function(){return Clock.pausetime||GetTime_() - Clock.gametime;}
*/
//----------------------------------------------------------------------------//

// Copy the original GetTime() function into a GetTime_() function (both now do the same)
// This is because clockengine.js  can modify the original GetTime() function to stop ticking.
if(typeof GetTime_ == 'undefined')
	GetTime_ = GetTime;

function ClockEngine(oldMsTime) {
	if (this instanceof ClockEngine == false)
		return new ClockEngine(oldtime);

	this.starttime= GetTime_(); //milliseconds sinds Sphere is running current session
	this.gametime=GetTime_() - (oldMsTime||0); //milliseconds of playing time to time int
	this.pausetime=0;
	this.seconds = 0;
	this.minutes = 0;
	this.hours = 0;
	this.delay = 0;
	this.redefineGetTime = true; // Redefine GetTime() to set a delay.
}

//----------------------------------------------------------------------------//

ClockEngine.prototype.start = function(oldMsTime) {this.gametime = GetTime_()-oldMsTime; return this;}

//----------------------------------------------------------------------------//

ClockEngine.prototype.pause = function(redefineGetTime) {
	this.pausetime = GetTime_();
	if(redefineGetTime === undefined) redefineGetTime = this.redefineGetTime;
	if(redefineGetTime)
		GetTime = Function("", "return "+ (GetTime_()-this.delay) );
	return this;
}

ClockEngine.prototype.unpause = function(redefineGetTime) {
	this.gametime += GetTime_()-this.pausetime;
	this.delay += GetTime_()-this.pausetime;
	this.pausetime = 0;
	if(redefineGetTime === undefined) redefineGetTime = this.redefineGetTime;
	if(this.redefineGetTime)
		GetTime = Function("", "return GetTime_()-"+ this.delay );
	return this;
}


/**
 * @retuns a Boolean, true if the game is paused, false if not
 */
ClockEngine.prototype.isPaused = function() {return (this.pausetime==0);}

//----------------------------------------------------------------------------//

/**
 * Returns a human readable time
 * @param {Boolean} returnAnArray False to return a string, true to return an array with [hours, minutes, seconds]
 * @param {Boolean} spheretime False to return how long Sphere has been running in this session, and true to return the gaming time
 * @returns a string or an object.
 * note: The values are also stored internally into the ClockEngine object: .hours, .minutes and .seconds
 */
ClockEngine.prototype.getTimeStr = function(spheretime) {
	this.calcTime(spheretime);
	return [this.hours, this.mins, this.secs];
}

ClockEngine.prototype.getTimeArr = function(spheretime) {
	this.calcTime(spheretime);
	return this.hours + ":" + this.mins + ":" + this.secs;
}

ClockEngine.prototype.calcTime = function(spheretime) {
	var s=Math.floor(( this.getMS(spheretime) )/1000);

	var n=s % 60;
	this.secs =((n<10)?'0':'') + n;

	var n=Math.floor(s/60) % 60;
	this.mins =((n<10)?'0':'') + n;

	var n=Math.floor(s/3600) % 60;
	this.hours =((n<100)?'0':'') + ((n<10)?'0':'') + n;
}

//----------------------------------------------------------------------------//
/**
 * Get The milliseconds you have been playing, without counting the time you paused the game.
 * @param {Boolean} spheretime False to return how long Sphere has been running in this session, and true to return the gaming time
 * @returns milliseconds of gaming time.
 */
ClockEngine.prototype.getMs = function(spheretime) {
	return (this.pausetime&&!spheretime?this.pausetime:GetTime_()) - (spheretime?this.starttime:this.gametime);
}

//----------------------------------------------------------------------------//
